﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Net;
using System.Net.Http;
using System.Net.Http.Headers;
using System.Net.Http.Json;
using System.Text.Json;
using System.Text.Json.Serialization;
using System.Threading.Tasks;
using AuthenticationServices;
using Foundation;
using UIKit;
using Xamarin.Essentials;

namespace test;

public partial class ViewController : UIViewController, IASWebAuthenticationPresentationContextProviding
{
    public UIWindow GetPresentationAnchor(ASWebAuthenticationSession session) =>
        (UIApplication.SharedApplication.Delegate as AppDelegate).Window;

    const string ClientId = "FILL_WITH_GOOGLE_CLIENT_ID";
    const string RedirectUri = $"com.googleusercontent.apps.{ClientId}:/oauth2redirect";
    const string FileName = "test.dat", FolderName = "XamarinTest";

    static HttpClient httpClient = new();

    public ViewController(IntPtr handle) : base(handle) { }

    public override async void ViewDidAppear(bool animated)
    {
        base.ViewDidAppear(animated);

        await Task.Delay(500);
        string accessToken = null;
        var authUri = new Uri("https://accounts.google.com/o/oauth2/v2/auth" +
            "?access_type=offline&response_type=code" +
            "&client_id=" + ClientId + ".apps.googleusercontent.com" +
            "&redirect_uri=" + WebUtility.UrlEncode(RedirectUri) +
            "&scope=" + WebUtility.UrlEncode("https://www.googleapis.com/auth/drive.file"));

        var callback = new TaskCompletionSource<Uri>();
        var session = new ASWebAuthenticationSession(authUri, new Uri(RedirectUri).Scheme, (callbackUrl, error) =>
        {
            if (error == null) { callback.TrySetResult(callbackUrl); }
            else { callback.TrySetException(new NSErrorException(error)); }
        });
        session.PresentationContextProvider = this;
        session.Start();
        var responseUrl = await callback.Task;
        var authResult = new WebAuthenticatorResult(responseUrl);

        var tokenUrl = "https://oauth2.googleapis.com/token";
        var _params = new Dictionary<string, string>()
        {
            ["grant_type"] = "authorization_code",
            ["client_id"] = ClientId,
            ["code"] = authResult.Get("code"),
            ["redirect_uri"] = RedirectUri,
        };
        var response = await httpClient.PostAsync(tokenUrl, new FormUrlEncodedContent(_params)).ConfigureAwait(false);
        var credentials = await response.Content.ReadFromJsonAsync<GoogleOAuthReponse>().ConfigureAwait(false);
        accessToken = credentials.access_token;

        var url = "https://www.googleapis.com/drive/v3/files?fields=files(id)&q=" +
            WebUtility.UrlEncode($"name='{FolderName}' and mimeType='application/vnd.google-apps.folder' and trashed=false and 'root' in parents");
        response = await SendRequest(HttpMethod.Get, url, accessToken).ConfigureAwait(false);
        var fileList = await response.Content.ReadFromJsonAsync<GoogleFileList>().ConfigureAwait(false);

        if (fileList.files.FirstOrDefault() is not GFile folder)
        {
            var content = JsonContent.Create(new GFile()
            {
                name = FolderName,
                mimeType = "application/vnd.google-apps.folder",
                parents = new[] { "root" }
            });
            response = await SendRequest(HttpMethod.Post, "https://www.googleapis.com/drive/v3/files", accessToken, content).ConfigureAwait(false);
            folder = await response.Content.ReadFromJsonAsync<GFile>().ConfigureAwait(false);
        }

        url = "https://www.googleapis.com/drive/v3/files?fields=files(id,name,size,properties)&q=" +
           WebUtility.UrlEncode($"name='{FileName}' and trashed=false and '{folder.id}' in parents");
        response = await SendRequest(HttpMethod.Get, url, accessToken).ConfigureAwait(false);
        var resultList = await response.Content.ReadFromJsonAsync<GoogleFileList>().ConfigureAwait(false);
        var file = resultList.files.FirstOrDefault();
        await UpdateFile(accessToken, file, null, new byte[213], FileName, folder.id);
    }

    static Task UpdateFile(string accessToken, GFile file, IDictionary<string, string> props, byte[] data, string fileName, string parentId)
    {
        HttpContent bodyContent = new StreamContent(new MemoryStream(data));
        var uploadType = props != null || file == null ? "multipart" : "media";
        if (uploadType == "multipart")
        {
            var databaseMeta = file != null ? new GFile() { properties = props } : new GFile() { name = fileName, properties = props, parents = new[] { parentId } };
            var metaDataContent = JsonContent.Create(databaseMeta, options: new JsonSerializerOptions { DefaultIgnoreCondition = JsonIgnoreCondition.WhenWritingDefault });
            metaDataContent.Headers.ContentDisposition = new ContentDispositionHeaderValue("form-data") { Name = "\"metadata\"" };
            bodyContent.Headers.ContentDisposition = new ContentDispositionHeaderValue("form-data") { Name = "\"file\"" };

            var boundary = Guid.NewGuid().ToString();
            bodyContent = new MultipartFormDataContent(boundary) { metaDataContent, bodyContent };
            bodyContent.Headers.Remove("Content-Type");
            bodyContent.Headers.TryAddWithoutValidation("Content-Type", "multipart/related; boundary=" + boundary);
        }

        if (file == null)
        {
            var url = $"https://www.googleapis.com/upload/drive/v3/files?uploadType={uploadType}&fields=";
            return SendRequest(HttpMethod.Post, url, accessToken, bodyContent);
        }
        else
        {
            var url = $"https://www.googleapis.com/upload/drive/v3/files/{file.id}?uploadType={uploadType}&fields=";
            return SendRequest(new HttpMethod("PATCH"), url, accessToken, bodyContent);
        }
    }

    public static async Task<HttpResponseMessage> SendRequest(HttpMethod method, string url, string token, HttpContent content = null)
    {
        var requestMessage = new HttpRequestMessage(method, url) { Content = content };
        requestMessage.Headers.Authorization = new AuthenticationHeaderValue("Bearer", token);
        var response = await httpClient.SendAsync(requestMessage).ConfigureAwait(false);
        if (!response.IsSuccessStatusCode)
        {
            var resultString = await response.Content.ReadAsStringAsync().ConfigureAwait(false);
            try { throw JsonSerializer.Deserialize<GoogleException>(resultString); }
            catch (JsonException) { throw new Exception(response.StatusCode + ": " + resultString); }
        }
        requestMessage.Dispose();
        return response;
    }
}

[Preserve(AllMembers = true)]
public class GFile
{
    public string id { get; set; }
    public string name { get; set; }
    public string mimeType { get; set; }
    public long? size { get; set; }
    public IList<string> parents { get; set; }
    public IDictionary<string, string> properties { get; set; }
}

[Preserve(AllMembers = true)]
public class GoogleException : Exception
{
    public GoogleRequestError error { get; set; }
    public override string Message => error?.code + ": " + error?.message;
}

[Preserve(AllMembers = true)]
public class GoogleRequestError
{
    public int code { get; set; }
    public string message { get; set; }
    public IList<SingleError> errors { get; set; }
}

[Preserve(AllMembers = true)]
public class SingleError
{
    public string domain { get; set; }
    public string reason { get; set; }
    public string message { get; set; }
}

[Preserve(AllMembers = true)]
public class GoogleOAuthReponse
{
    public string access_token { get; set; }
}

[Preserve(AllMembers = true)]
public class GoogleFileList
{
    public IList<GFile> files { get; set; }
}