package com.example.demo;

import org.springframework.batch.core.configuration.annotation.StepScope;
import org.springframework.batch.core.job.Job;
import org.springframework.batch.core.job.builder.JobBuilder;
import org.springframework.batch.core.partition.PartitionHandler;
import org.springframework.batch.core.partition.Partitioner;
import org.springframework.batch.core.partition.support.TaskExecutorPartitionHandler;
import org.springframework.batch.core.repository.JobRepository;
import org.springframework.batch.core.step.Step;
import org.springframework.batch.core.step.builder.StepBuilder;
import org.springframework.batch.infrastructure.item.ExecutionContext;
import org.springframework.batch.infrastructure.item.ItemReader;
import org.springframework.batch.infrastructure.item.ItemWriter;
import org.springframework.batch.infrastructure.item.support.ListItemReader;
import org.springframework.batch.infrastructure.support.transaction.ResourcelessTransactionManager;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.task.TaskExecutor;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.transaction.PlatformTransactionManager;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Configuration
public class JobConfig {
    private static final String DISCRIMINATOR_KEY = "DISCRIMINATOR";

    @Bean
    public PlatformTransactionManager platformTransactionManager() {
        return new ResourcelessTransactionManager();
    }

    @Bean
    public Job job(JobRepository jobRepository,
                   Step partitionStep) {

        return new JobBuilder("partitionJob", jobRepository)
                .start(partitionStep)
                .build();
    }

    @Bean
    public Step partitionStep(JobRepository jobRepository,
                              PartitionHandler partitionHandler,
                              Partitioner partitioner) {
        return new StepBuilder("partitionCommonStep", jobRepository)
                .partitioner("partitioner", partitioner)
                .partitionHandler(partitionHandler)
                .build();
    }

    @Bean
    public TaskExecutorPartitionHandler partitionHandler(Step workStep,
                                                         TaskExecutor batchTaskExecutor) {
        TaskExecutorPartitionHandler partitionHandler = new TaskExecutorPartitionHandler();
        partitionHandler.setTaskExecutor(batchTaskExecutor);
        partitionHandler.setStep(workStep);
        partitionHandler.setGridSize(3);
        return partitionHandler;
    }

    @Bean
    public Step workStep(JobRepository jobRepository,
                         ItemReader<Integer> itemReader,
                         ItemWriter<Integer> itemWriter,
                         PlatformTransactionManager platformTransactionManager) {

        return new StepBuilder("workStep", jobRepository)
//                .<Integer, Integer>chunk(3, platformTransactionManager) // WORKS
                .<Integer, Integer>chunk(3)
                .reader(itemReader)
                .processor(item -> item)
                .writer(itemWriter)
                .build();
    }

    @Bean
    public ThreadPoolTaskExecutor batchTaskExecutor() {
        ThreadPoolTaskExecutor taskExecutor = new ThreadPoolTaskExecutor();
        taskExecutor.setCorePoolSize(5);
        taskExecutor.setQueueCapacity(Integer.MAX_VALUE);
        taskExecutor.setMaxPoolSize(Integer.MAX_VALUE);
        taskExecutor.setDaemon(true);
        return taskExecutor;
    }

    @Bean
    public Partitioner partitioner() {
        return gridSize -> {
            Map<String, ExecutionContext> result = new HashMap<>();

            for (int i = 1; i <= 3; i++) {
                ExecutionContext value = new ExecutionContext();
                result.put("partition" + i, value);
                value.put(DISCRIMINATOR_KEY, i);
            }

            return result;
        };
    }

    @Bean
    @StepScope
    public List<Integer> items(@Value("#{stepExecutionContext['" + DISCRIMINATOR_KEY + "']}") String partition) {
        ArrayList<Integer> items = new ArrayList<>();
        int itemCount = Integer.parseInt(partition) * 10;
        for (int i = 0; i < itemCount; i++) {
            items.add(i);
        }
        System.out.printf("items: %d\n", itemCount);
        return items;
    }

    @Bean
    public ItemReader<Integer> reader(List<Integer> items) {
        return new ListItemReader<>(items);
    }

    @Bean
    public ItemWriter<Integer> writer() {
        return chunk -> chunk.getItems()
                .forEach(System.out::println);
    }
}
