# %%
from dotenv import load_dotenv
from nautilus_trader.adapters.interactive_brokers.common import IB_VENUE
from nautilus_trader.backtest.engine import BacktestEngine
from nautilus_trader.backtest.models import (
    BestPriceFillModel,
    FillModel,
    OneTickSlippageFillModel,
)
from nautilus_trader.common.enums import LogColor
from nautilus_trader.config import BacktestEngineConfig, LoggingConfig
from nautilus_trader.execution.config import ExecEngineConfig
from nautilus_trader.model.currencies import USD
from nautilus_trader.model.data import Bar, BarType
from nautilus_trader.model.enums import (
    AccountType,
    OmsType,
    OrderSide,
    PositionSide,
    TimeInForce,
)
from nautilus_trader.model.events import PositionClosed, PositionOpened
from nautilus_trader.model.identifiers import InstrumentId, TraderId, Venue
from nautilus_trader.model.objects import Money
from nautilus_trader.persistence.catalog.parquet import ParquetDataCatalog
from nautilus_trader.trading.config import StrategyConfig
from nautilus_trader.trading.strategy import Strategy


# %%
class DemoStrategyConfig(StrategyConfig, frozen=True):
    bar_type: BarType
    instrument_id: InstrumentId


class DemoStrategy(Strategy):
    def __init__(self, config: DemoStrategyConfig):
        super().__init__(config=config)

        # Track total bars seen
        self.count_of_bars: int = 0
        self.show_portfolio_at_bar: int | None = 0
        self.in_position = False
        self.order_placed = False

    def on_start(self):
        """
        Handle strategy start event.
        """
        # self.request_instrument(self.config.instrument_id)
        self.instrument = self.cache.instrument(self.config.instrument_id)
        self.subscribe_bars(self.config.bar_type)

    def on_bar(self, bar: Bar):
        """
        Handle new bar event.
        """
        # Increment total bars seen
        self.count_of_bars += 1
        self.log.warning(f"on_bar, bar number {self.count_of_bars}")

        if (not self.in_position) and (not self.order_placed):

            order = self.order_factory.market(
                instrument_id=self.config.instrument_id,
                order_side=OrderSide.BUY,
                quantity=self.instrument.make_qty(100),
                time_in_force=TimeInForce.GTC,
            )

            self.submit_order(order)
            self.order_placed = True
            self.log.info(f"Submitted market order: {order}", color=LogColor.GREEN)

        elif (self.in_position) and (not self.order_placed):

            order = self.order_factory.market(
                instrument_id=self.config.instrument_id,
                order_side=OrderSide.SELL,
                quantity=self.instrument.make_qty(100),
                time_in_force=TimeInForce.GTC,
            )

            self.submit_order(order)
            self.order_placed = True
            self.log.info(f"Submitted market order: {order}", color=LogColor.GREEN)

            return

    def on_order_filled(self, event):

        # Log position details
        self.log.warning(f"Order filled: {event}", color=LogColor.GREEN)
        self.order_placed = False

    def on_position_opened(self, event: PositionOpened):
        """
        Handle position opened event.
        """
        # Log position details
        self.log.warning(f"Position opened: {event}", color=LogColor.GREEN)
        self.in_position = True

    def on_position_closed(self, event: PositionClosed):
        """
        Handle position opened event.
        """
        # Log position details
        self.log.warning(f"Position closed: {event}", color=LogColor.GREEN)
        self.in_position = False

    def on_stop(self):
        """
        Handle strategy stop event.
        """
        # close all positions
        self.close_all_positions(self.instrument.id, position_side=PositionSide.LONG)


# %%
# 1. DEFINE THE ENGINE
######################
# set up run specific names for the log and the data folder
# kepp the number of such files under control
trader_id = "TEST_BACKTEST-001"
engine_config = BacktestEngineConfig(
    trader_id=TraderId(trader_id),
    logging=LoggingConfig(
        log_level="WARNING",
        log_level_file="DEBUG",
        log_directory="log",
        log_file_max_size=50_000_000,
        log_file_name=f"{trader_id}",  # _{dt.datetime.now(dt.UTC).strftime("%Y-%m-%d_%H%M%S")}.log",
        # log_component_levels={"ExecEngine": "WARNING"},
    ),
    cache=None,  # don't use the cache because it distrupts loading of instruments
    exec_engine=ExecEngineConfig(debug=True),
)
engine = BacktestEngine(config=engine_config)

# fill_model = FillModel(
#     prob_fill_on_limit=1,  # Chance a limit order fills when price matches (applied to bars/trades/quotes + L1/L2/L3 order book)
#     prob_slippage=1,  # MAX VALUE: Chance of 1-tick slippage (applied to bars/trades/quotes + L1 order book only)
#     random_seed=42,  # Optional: Set for reproducible results
# )
fill_model = OneTickSlippageFillModel()
# %%
catalog = ParquetDataCatalog("data/nautilus/catalog")
instruments = catalog.instruments()
instrument = list(set([i for i in instruments if str(i.raw_symbol) == "B"]))[0]

venue = Venue("NYSE")
engine.add_venue(
    venue=venue,
    oms_type=OmsType.NETTING,
    account_type=AccountType.CASH,  # Spot CASH account (not for perpetuals or futures)
    base_currency=USD,
    starting_balances=[Money(1_000_000, USD)],
    fill_model=fill_model,
)


engine.add_instrument(instrument)
# data
bar_type = BarType.from_str(f"{instrument.id}-1-HOUR-LAST-EXTERNAL")
bars_data = catalog.bars(
    bar_types=[str(bar_type)],
)
engine.add_data(bars_data)

strategy_config = DemoStrategyConfig(
    # bar_type=BarType.from_str(f"{instrument_id}-1-MINUTE-LAST-EXTERNAL"),
    bar_type=bar_type,
    instrument_id=instrument.id,
)
strategy = DemoStrategy(config=strategy_config)
engine.add_strategy(strategy=strategy)


engine.run()
